/*
 * Decompiled with CFR 0.152.
 */
package com.crittafur.statsapi;

import com.crittafur.statsapi.AdvancementsReader;
import com.crittafur.statsapi.StatsParser;
import com.crittafur.statsapi.StatsUtils;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.server.players.GameProfileCache;
import net.minecraft.world.level.storage.LevelResource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StatsReader {
    private static final Logger LOGGER = LoggerFactory.getLogger((String)"StatsAPI");
    private final MinecraftServer server;
    private final Path statsDirectory;
    private final AdvancementsReader advancementsReader;

    public StatsReader(MinecraftServer server, AdvancementsReader advancementsReader) {
        this.server = server;
        this.advancementsReader = advancementsReader;
        this.statsDirectory = server.getWorldPath(LevelResource.PLAYER_STATS_DIR);
        LOGGER.info("[Reader] Stats directory: {}", (Object)this.statsDirectory);
    }

    public Map<String, Object> getPlayerStats(String uuidString) {
        String normalizedUuid = StatsUtils.normalizeUuid(uuidString);
        Path statsFile = this.statsDirectory.resolve(normalizedUuid + ".json");
        if (!Files.exists(statsFile, new LinkOption[0])) {
            LOGGER.debug("[Reader] Stats file not found for UUID: {}", (Object)normalizedUuid);
            return null;
        }
        try {
            String playerName = this.getPlayerName(normalizedUuid);
            return StatsParser.parseStatsFile(statsFile, normalizedUuid, playerName);
        }
        catch (IOException e) {
            LOGGER.error("[Reader] Failed to read stats for {}", (Object)normalizedUuid, (Object)e);
            return null;
        }
    }

    public Map<String, Map<String, Object>> getAllPlayerStats() {
        LinkedHashMap<String, Map<String, Object>> allStats = new LinkedHashMap<String, Map<String, Object>>();
        try (Stream<Path> files = Files.list(this.statsDirectory);){
            files.filter(p -> p.toString().endsWith(".json")).forEach(statsFile -> {
                String uuid = statsFile.getFileName().toString().replace(".json", "");
                try {
                    String playerName = this.getPlayerName(uuid);
                    Map<String, Object> stats = StatsParser.parseStatsFile(statsFile, uuid, playerName);
                    if (stats != null) {
                        allStats.put(uuid, stats);
                    }
                }
                catch (IOException e) {
                    LOGGER.error("[Reader] Failed to read stats file: {}", statsFile, (Object)e);
                }
            });
        }
        catch (IOException e) {
            LOGGER.error("[Reader] Failed to list stats directory", (Throwable)e);
        }
        return allStats;
    }

    public Map<String, Object> getServerStats() {
        return this.getServerStats(this.getAllPlayerStats());
    }

    public Map<String, Object> getServerStats(Map<String, Map<String, Object>> allStats) {
        return StatsParser.aggregateServerStats(allStats.values());
    }

    public List<Map<String, Object>> getLeaderboard(String statName, int limit) {
        Map<String, Map<String, Object>> allStats = this.getAllPlayerStats();
        return allStats.entrySet().stream().map(entry -> {
            Map stats = (Map)entry.getValue();
            LinkedHashMap<String, Object> leaderboardEntry = new LinkedHashMap<String, Object>();
            leaderboardEntry.put("uuid", entry.getKey());
            leaderboardEntry.put("name", stats.get("name"));
            long value = this.getStatValue(stats, statName);
            leaderboardEntry.put("value", value);
            String formatted = this.formatStatValue(statName, value);
            if (formatted != null) {
                leaderboardEntry.put("formatted", formatted);
            }
            return leaderboardEntry;
        }).sorted((a, b) -> Long.compare(((Number)b.get("value")).longValue(), ((Number)a.get("value")).longValue())).limit(limit).collect(Collectors.toList());
    }

    private long getStatValue(Map<String, Object> stats, String statName) {
        switch (statName.toLowerCase()) {
            case "advancements_completed": {
                Map<String, Object> advancements;
                String uuid = (String)stats.get("uuid");
                if (uuid != null && this.advancementsReader != null && (advancements = this.advancementsReader.getPlayerAdvancements(uuid, false)) != null) {
                    Number val = (Number)advancements.get("completed");
                    return val != null ? val.longValue() : 0L;
                }
                return 0L;
            }
            case "blocks_mined": 
            case "total_blocks_mined": {
                Map mining = (Map)stats.get("mining");
                if (mining != null) {
                    Number val = (Number)mining.get("total_blocks");
                    return val != null ? val.longValue() : 0L;
                }
                return 0L;
            }
            case "mob_kills": 
            case "kills": 
            case "total_mob_kills": {
                Map combat = (Map)stats.get("combat");
                if (combat != null) {
                    Number val = (Number)combat.get("kills");
                    return val != null ? val.longValue() : 0L;
                }
                return 0L;
            }
            case "items_crafted": 
            case "total_items_crafted": {
                Map crafting = (Map)stats.get("items_crafted");
                if (crafting != null) {
                    Number val = (Number)crafting.get("total_items");
                    return val != null ? val.longValue() : 0L;
                }
                return 0L;
            }
            case "items_used": 
            case "total_items_used": {
                Map itemsUsed = (Map)stats.get("items_used");
                if (itemsUsed != null) {
                    Number val = (Number)itemsUsed.get("total");
                    return val != null ? val.longValue() : 0L;
                }
                return 0L;
            }
            case "distance": 
            case "total_distance": {
                Map distance = (Map)stats.get("distance");
                if (distance != null) {
                    Number val = (Number)distance.get("total_km");
                    return val != null ? (long)(val.doubleValue() * 100000.0) : 0L;
                }
                return 0L;
            }
            case "walking_distance": {
                Map distance = (Map)stats.get("distance");
                if (distance != null) {
                    Number val = (Number)distance.get("walking_km");
                    return val != null ? (long)(val.doubleValue() * 100000.0) : 0L;
                }
                return 0L;
            }
            case "sprinting_distance": {
                Map distance = (Map)stats.get("distance");
                if (distance != null) {
                    Number val = (Number)distance.get("sprinting_km");
                    return val != null ? (long)(val.doubleValue() * 100000.0) : 0L;
                }
                return 0L;
            }
            case "swimming_distance": {
                Map distance = (Map)stats.get("distance");
                if (distance != null) {
                    Number val = (Number)distance.get("swimming_km");
                    return val != null ? (long)(val.doubleValue() * 100000.0) : 0L;
                }
                return 0L;
            }
            case "flying_distance": {
                Map distance = (Map)stats.get("distance");
                if (distance != null) {
                    Number val = (Number)distance.get("flying_km");
                    return val != null ? (long)(val.doubleValue() * 100000.0) : 0L;
                }
                return 0L;
            }
            case "elytra_distance": {
                Map distance = (Map)stats.get("distance");
                if (distance != null) {
                    Number val = (Number)distance.get("elytra_km");
                    return val != null ? (long)(val.doubleValue() * 100000.0) : 0L;
                }
                return 0L;
            }
            case "horse_distance": {
                Map distance = (Map)stats.get("distance");
                if (distance != null) {
                    Number val = (Number)distance.get("horse_km");
                    return val != null ? (long)(val.doubleValue() * 100000.0) : 0L;
                }
                return 0L;
            }
            case "boating_distance": {
                Map distance = (Map)stats.get("distance");
                if (distance != null) {
                    Number val = (Number)distance.get("boating_km");
                    return val != null ? (long)(val.doubleValue() * 100000.0) : 0L;
                }
                return 0L;
            }
            case "player_kills": {
                Map combat = (Map)stats.get("combat");
                if (combat != null) {
                    Number val = (Number)combat.get("player_kills");
                    return val != null ? val.longValue() : 0L;
                }
                return 0L;
            }
            case "deaths": {
                Map combat = (Map)stats.get("combat");
                if (combat != null) {
                    Number val = (Number)combat.get("deaths");
                    return val != null ? val.longValue() : 0L;
                }
                return 0L;
            }
            case "damage_dealt": {
                Map combat = (Map)stats.get("combat");
                if (combat != null) {
                    Number val = (Number)combat.get("damage_dealt");
                    return val != null ? val.longValue() : 0L;
                }
                return 0L;
            }
            case "damage_taken": {
                Map combat = (Map)stats.get("combat");
                if (combat != null) {
                    Number val = (Number)combat.get("damage_taken");
                    return val != null ? val.longValue() : 0L;
                }
                return 0L;
            }
            case "chests_opened": {
                Map interactions = (Map)stats.get("interactions");
                if (interactions != null) {
                    Number val = (Number)interactions.get("chests_opened");
                    return val != null ? val.longValue() : 0L;
                }
                return 0L;
            }
            case "beds_slept_in": {
                Map interactions = (Map)stats.get("interactions");
                if (interactions != null) {
                    Number val = (Number)interactions.get("beds_slept_in");
                    return val != null ? val.longValue() : 0L;
                }
                return 0L;
            }
            case "containers_looted": {
                Map interactions = (Map)stats.get("interactions");
                if (interactions != null) {
                    Number val = (Number)interactions.get("containers_looted");
                    return val != null ? val.longValue() : 0L;
                }
                return 0L;
            }
            case "tools_broken": {
                Map toolsBroken = (Map)stats.get("tools_broken");
                if (toolsBroken != null) {
                    Number val = (Number)toolsBroken.get("total");
                    return val != null ? val.longValue() : 0L;
                }
                return 0L;
            }
        }
        Map custom = (Map)stats.get("custom");
        if (custom == null) {
            return 0L;
        }
        String statKey = this.mapStatName(statName);
        Number value = (Number)custom.get(statKey);
        return value != null ? value.longValue() : 0L;
    }

    private String formatStatValue(String statName, long value) {
        return switch (statName.toLowerCase()) {
            case "play_time", "playtime", "time" -> StatsUtils.formatPlaytime(value);
            case "distance", "total_distance", "walking_distance", "sprinting_distance", "swimming_distance", "flying_distance", "elytra_distance", "horse_distance", "boating_distance" -> this.formatDistance(value);
            case "damage_dealt", "damage_taken" -> this.formatDamage(value);
            default -> null;
        };
    }

    private String formatDistance(long cm) {
        double blocks = (double)cm / 100.0;
        if (blocks >= 1000.0) {
            return String.format("%.1fkm", blocks / 1000.0);
        }
        return String.format("%.0f blocks", blocks);
    }

    private String formatDamage(long raw) {
        double hearts = (double)raw / 20.0;
        return String.format("%.1f hearts", hearts);
    }

    private String mapStatName(String friendlyName) {
        return switch (friendlyName.toLowerCase()) {
            case "play_time", "playtime", "time" -> "play_time";
            case "jumps", "jump" -> "jump";
            default -> friendlyName;
        };
    }

    public Map<String, Map<String, Object>> getRanksForPlayer(String uuidString, Set<String> validStats) {
        return this.getRanksForPlayer(uuidString, validStats, this.getAllPlayerStats());
    }

    public Map<String, Map<String, Object>> getRanksForPlayer(String uuidString, Set<String> validStats, Map<String, Map<String, Object>> allStats) {
        String normalizedUuid = StatsUtils.normalizeUuid(uuidString);
        if (!allStats.containsKey(normalizedUuid)) {
            return null;
        }
        Map<String, Object> playerStats = allStats.get(normalizedUuid);
        LinkedHashMap<String, Map<String, Object>> ranks = new LinkedHashMap<String, Map<String, Object>>();
        for (String statName : validStats) {
            long playerValue = this.getStatValue(playerStats, statName);
            if (playerValue <= 0L) continue;
            int rank = 1;
            int total = 0;
            for (Map<String, Object> otherStats : allStats.values()) {
                long otherValue = this.getStatValue(otherStats, statName);
                if (otherValue <= 0L) continue;
                ++total;
                if (otherValue <= playerValue) continue;
                ++rank;
            }
            if (total <= 0) continue;
            LinkedHashMap<String, Integer> rankInfo = new LinkedHashMap<String, Integer>();
            rankInfo.put("rank", rank);
            rankInfo.put("total", total);
            ranks.put(statName, rankInfo);
        }
        return ranks;
    }

    public String getPlayerName(String uuid) {
        try {
            UUID playerUuid = UUID.fromString(uuid);
            ServerPlayer player = this.server.getPlayerList().getPlayer(playerUuid);
            if (player != null) {
                return player.getName().getString();
            }
            GameProfileCache cache = this.server.getProfileCache();
            if (cache != null) {
                return cache.get(playerUuid).map(profile -> profile.getName()).orElse("Unknown");
            }
        }
        catch (IllegalArgumentException e) {
            LOGGER.debug("[Reader] Invalid UUID format: {}", (Object)uuid);
        }
        return "Unknown";
    }
}

