/*
 * Decompiled with CFR 0.152.
 */
package com.crittafur.statsapi;

import com.crittafur.statsapi.StatsUtils;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.google.gson.JsonSyntaxException;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StatsParser {
    private static final Logger LOGGER = LoggerFactory.getLogger((String)"StatsAPI");
    private static final long MAX_STATS_FILE_SIZE = 0xA00000L;
    private static final int TICKS_PER_SECOND = 20;
    private static final int TICKS_PER_HOUR = 72000;

    public static Map<String, Object> parseStatsFile(Path file, String uuid, String playerName) throws IOException {
        long fileSize = Files.size(file);
        if (fileSize > 0xA00000L) {
            LOGGER.warn("[Parser] Stats file too large, skipping: {} ({} bytes)", (Object)file, (Object)fileSize);
            return null;
        }
        String content = Files.readString(file);
        return StatsParser.parseStatsJson(content, uuid, playerName);
    }

    public static Map<String, Object> parseStatsJson(String jsonContent, String uuid, String playerName) {
        JsonElement parsed;
        try {
            parsed = JsonParser.parseString((String)jsonContent);
        }
        catch (JsonSyntaxException e) {
            LOGGER.warn("[Parser] Invalid JSON syntax: {}", (Object)e.getMessage());
            return null;
        }
        if (!parsed.isJsonObject()) {
            LOGGER.warn("[Parser] Stats content is not a JSON object");
            return null;
        }
        JsonObject root = parsed.getAsJsonObject();
        if (!root.has("stats")) {
            return null;
        }
        LinkedHashMap<String, Object> result = new LinkedHashMap<String, Object>();
        result.put("uuid", uuid);
        result.put("name", playerName);
        JsonObject stats = root.getAsJsonObject("stats");
        if (stats.has("minecraft:custom")) {
            result.put("custom", StatsParser.extractCategory(stats.getAsJsonObject("minecraft:custom")));
        }
        if (stats.has("minecraft:mined")) {
            result.put("mined", StatsParser.extractCategory(stats.getAsJsonObject("minecraft:mined")));
        }
        if (stats.has("minecraft:killed")) {
            result.put("killed", StatsParser.extractCategory(stats.getAsJsonObject("minecraft:killed")));
        }
        if (stats.has("minecraft:killed_by")) {
            result.put("killed_by", StatsParser.extractCategory(stats.getAsJsonObject("minecraft:killed_by")));
        }
        if (stats.has("minecraft:used")) {
            result.put("used", StatsParser.extractCategory(stats.getAsJsonObject("minecraft:used")));
        }
        if (stats.has("minecraft:crafted")) {
            result.put("crafted", StatsParser.extractCategory(stats.getAsJsonObject("minecraft:crafted")));
        }
        if (stats.has("minecraft:picked_up")) {
            result.put("picked_up", StatsParser.extractCategory(stats.getAsJsonObject("minecraft:picked_up")));
        }
        if (stats.has("minecraft:dropped")) {
            result.put("dropped", StatsParser.extractCategory(stats.getAsJsonObject("minecraft:dropped")));
        }
        if (stats.has("minecraft:broken")) {
            result.put("broken", StatsParser.extractCategory(stats.getAsJsonObject("minecraft:broken")));
        }
        StatsParser.addComputedStats(result);
        return result;
    }

    static Map<String, Object> extractCategory(JsonObject category) {
        LinkedHashMap<String, Object> result = new LinkedHashMap<String, Object>();
        for (Map.Entry entry : category.entrySet()) {
            String key = (String)entry.getKey();
            int colonIndex = key.indexOf(58);
            if (colonIndex >= 0) {
                key = key.substring(colonIndex + 1);
            }
            if (key.equals("air")) continue;
            JsonElement value = (JsonElement)entry.getValue();
            if (value.isJsonPrimitive() && value.getAsJsonPrimitive().isNumber()) {
                result.put(key, value.getAsLong());
                continue;
            }
            LOGGER.debug("[Parser] Skipping non-numeric stat: {} = {}", (Object)key, (Object)value);
        }
        return result;
    }

    static void addComputedStats(Map<String, Object> stats) {
        Map dropped;
        Map pickedUp;
        Map broken;
        Map used;
        Map crafted;
        Map mined;
        Map custom = (Map)stats.get("custom");
        if (custom != null && custom.containsKey("play_time")) {
            long ticks = ((Number)custom.get("play_time")).longValue();
            stats.put("playtime_formatted", StatsUtils.formatPlaytime(ticks));
            stats.put("playtime_hours", (double)Math.round((double)ticks / 72000.0 * 10.0) / 10.0);
        }
        if (custom != null) {
            Map<String, Object> distance = StatsParser.buildDistanceSection(custom);
            stats.put("distance", distance);
        }
        Map<String, Object> combat = StatsParser.buildCombatSection(custom, (Map)stats.get("killed"));
        stats.put("combat", combat);
        if (custom != null) {
            Map<String, Object> interactions = StatsParser.buildInteractionsSection(custom);
            stats.put("interactions", interactions);
        }
        if ((mined = (Map)stats.get("mined")) != null) {
            Map<String, Object> mining = StatsParser.buildMiningSection(mined);
            stats.put("mining", mining);
        }
        if ((crafted = (Map)stats.get("crafted")) != null) {
            Map<String, Object> crafting = StatsParser.buildCraftingSection(crafted);
            stats.put("items_crafted", crafting);
        }
        if ((used = (Map)stats.get("used")) != null) {
            Map<String, Object> itemsUsed = StatsParser.buildItemsUsedSection(used);
            stats.put("items_used", itemsUsed);
        }
        if ((broken = (Map)stats.get("broken")) != null) {
            Map<String, Object> toolsBroken = StatsParser.buildToolsBrokenSection(broken);
            stats.put("tools_broken", toolsBroken);
        }
        if ((pickedUp = (Map)stats.get("picked_up")) != null) {
            long totalPickedUp = pickedUp.values().stream().mapToLong(v -> ((Number)v).longValue()).sum();
            stats.put("items_picked_up", totalPickedUp);
        }
        if ((dropped = (Map)stats.get("dropped")) != null) {
            long totalDropped = dropped.values().stream().mapToLong(v -> ((Number)v).longValue()).sum();
            stats.put("items_dropped", totalDropped);
        }
    }

    private static Map<String, Object> buildDistanceSection(Map<String, Object> custom) {
        LinkedHashMap<String, Double> distance = new LinkedHashMap<String, Double>();
        String[][] mappings = new String[][]{{"walking_km", "walk_one_cm"}, {"sprinting_km", "sprint_one_cm"}, {"swimming_km", "swim_one_cm"}, {"flying_km", "fly_one_cm"}, {"falling_km", "fall_one_cm"}, {"climbing_km", "climb_one_cm"}, {"crouching_km", "crouch_one_cm"}, {"walking_on_water_km", "walk_on_water_one_cm"}, {"walking_underwater_km", "walk_under_water_one_cm"}, {"boating_km", "boat_one_cm"}, {"minecart_km", "minecart_one_cm"}, {"horse_km", "horse_one_cm"}, {"pig_km", "pig_one_cm"}, {"strider_km", "strider_one_cm"}, {"elytra_km", "aviate_one_cm"}};
        long totalCm = 0L;
        for (String[] mapping : mappings) {
            String outputKey = mapping[0];
            String rawKey = mapping[1];
            Number val = (Number)custom.get(rawKey);
            long cm = val != null ? val.longValue() : 0L;
            totalCm += cm;
            if (cm <= 0L) continue;
            distance.put(outputKey, StatsParser.cmToKm(cm));
        }
        LinkedHashMap<String, Object> result = new LinkedHashMap<String, Object>();
        result.put("total_km", StatsParser.cmToKm(totalCm));
        result.putAll(distance);
        return result;
    }

    private static Map<String, Object> buildCombatSection(Map<String, Object> custom, Map<String, Object> killed) {
        LinkedHashMap<String, Object> combat = new LinkedHashMap<String, Object>();
        long mobKills = custom != null ? StatsParser.getLong(custom, "mob_kills") : 0L;
        long playerKills = custom != null ? StatsParser.getLong(custom, "player_kills") : 0L;
        long deaths = custom != null ? StatsParser.getLong(custom, "deaths") : 0L;
        long damageDealt = custom != null ? StatsParser.getLong(custom, "damage_dealt") : 0L;
        long damageTaken = custom != null ? StatsParser.getLong(custom, "damage_taken") : 0L;
        combat.put("kills", mobKills);
        combat.put("player_kills", playerKills);
        combat.put("deaths", deaths);
        combat.put("damage_dealt", damageDealt);
        combat.put("damage_dealt_hearts", (double)Math.round((double)damageDealt / 20.0 * 10.0) / 10.0);
        combat.put("damage_taken", damageTaken);
        combat.put("damage_taken_hearts", (double)Math.round((double)damageTaken / 20.0 * 10.0) / 10.0);
        if (killed != null && !killed.isEmpty()) {
            combat.put("unique_mobs_killed", killed.size());
            combat.put("top_mobs_killed", StatsParser.extractTopN(killed, "mob", 5));
        }
        return combat;
    }

    private static Map<String, Object> buildInteractionsSection(Map<String, Object> custom) {
        String[][] mappings;
        LinkedHashMap<String, Object> interactions = new LinkedHashMap<String, Object>();
        for (String[] mapping : mappings = new String[][]{{"chests_opened", "open_chest"}, {"furnaces_used", "interact_with_furnace"}, {"crafting_tables_used", "interact_with_crafting_table"}, {"beds_slept_in", "sleep_in_bed"}, {"villagers_talked_to", "talked_to_villager"}, {"enchanting_tables_used", "interact_with_enchanting_table"}, {"anvils_used", "interact_with_anvil"}, {"beacons_activated", "interact_with_beacon"}, {"noteblocks_played", "play_noteblock"}, {"flower_pots_filled", "pot_flower"}, {"containers_looted", "looted_stat"}}) {
            interactions.put(mapping[0], StatsParser.getLong(custom, mapping[1]));
        }
        return interactions;
    }

    private static Map<String, Object> buildMiningSection(Map<String, Object> mined) {
        LinkedHashMap<String, Object> mining = new LinkedHashMap<String, Object>();
        long totalMined = mined.values().stream().mapToLong(v -> ((Number)v).longValue()).sum();
        mining.put("total_blocks", totalMined);
        mining.put("unique_blocks", mined.size());
        mining.put("top_blocks", StatsParser.extractTopN(mined, "block", 5));
        return mining;
    }

    private static Map<String, Object> buildCraftingSection(Map<String, Object> crafted) {
        LinkedHashMap<String, Object> crafting = new LinkedHashMap<String, Object>();
        long totalCrafted = crafted.values().stream().mapToLong(v -> ((Number)v).longValue()).sum();
        crafting.put("total_items", totalCrafted);
        crafting.put("unique_items", crafted.size());
        crafting.put("top_items", StatsParser.extractTopN(crafted, "item", 5));
        return crafting;
    }

    private static Map<String, Object> buildItemsUsedSection(Map<String, Object> used) {
        LinkedHashMap<String, Object> itemsUsed = new LinkedHashMap<String, Object>();
        long totalUsed = used.values().stream().mapToLong(v -> ((Number)v).longValue()).sum();
        itemsUsed.put("total", totalUsed);
        itemsUsed.put("top_items", StatsParser.extractTopN(used, "item", 5));
        return itemsUsed;
    }

    private static Map<String, Object> buildToolsBrokenSection(Map<String, Object> broken) {
        LinkedHashMap<String, Object> toolsBroken = new LinkedHashMap<String, Object>();
        long totalBroken = broken.values().stream().mapToLong(v -> ((Number)v).longValue()).sum();
        toolsBroken.put("total", totalBroken);
        toolsBroken.put("top_items", StatsParser.extractTopN(broken, "item", 5));
        return toolsBroken;
    }

    static List<Map<String, Object>> extractTopN(Map<String, Object> source, String keyName, int limit) {
        return source.entrySet().stream().sorted((a, b) -> Long.compare(((Number)b.getValue()).longValue(), ((Number)a.getValue()).longValue())).limit(limit).map(entry -> {
            LinkedHashMap<String, Object> item = new LinkedHashMap<String, Object>();
            item.put(keyName, entry.getKey());
            item.put("count", entry.getValue());
            return item;
        }).collect(Collectors.toList());
    }

    private static double cmToKm(long cm) {
        return (double)Math.round((double)cm / 100000.0 * 100.0) / 100.0;
    }

    private static long getLong(Map<String, Object> map, String key) {
        Number val = (Number)map.get(key);
        return val != null ? val.longValue() : 0L;
    }

    public static Map<String, Object> aggregateServerStats(Collection<Map<String, Object>> allPlayerStats) {
        String[] distanceKeys;
        LinkedHashMap<String, Object> result = new LinkedHashMap<String, Object>();
        result.put("unique_players", allPlayerStats.size());
        long totalPlaytimeTicks = 0L;
        LinkedHashMap<String, Long> distanceTotals = new LinkedHashMap<String, Long>();
        for (String key : distanceKeys = new String[]{"walk_one_cm", "sprint_one_cm", "swim_one_cm", "fly_one_cm", "fall_one_cm", "climb_one_cm", "crouch_one_cm", "walk_on_water_one_cm", "walk_under_water_one_cm", "boat_one_cm", "minecart_one_cm", "horse_one_cm", "pig_one_cm", "strider_one_cm", "aviate_one_cm"}) {
            distanceTotals.put(key, 0L);
        }
        long totalKills = 0L;
        long totalPlayerKills = 0L;
        long totalDeaths = 0L;
        long totalDamageDealt = 0L;
        long totalDamageTaken = 0L;
        LinkedHashMap<String, Long> killedAggregated = new LinkedHashMap<String, Long>();
        LinkedHashMap<String, Long> minedAggregated = new LinkedHashMap<String, Long>();
        LinkedHashMap<String, Long> craftedAggregated = new LinkedHashMap<String, Long>();
        LinkedHashMap<String, Long> usedAggregated = new LinkedHashMap<String, Long>();
        LinkedHashMap<String, Long> brokenAggregated = new LinkedHashMap<String, Long>();
        LinkedHashMap<String, Long> interactionTotals = new LinkedHashMap<String, Long>();
        String[] interactionKeys = new String[]{"open_chest", "interact_with_furnace", "interact_with_crafting_table", "sleep_in_bed", "talked_to_villager", "looted_stat"};
        for (String string : interactionKeys) {
            interactionTotals.put(string, 0L);
        }
        for (Map map : allPlayerStats) {
            Map broken;
            Map map2;
            Map crafted;
            Map mined;
            Map map3;
            Map custom = (Map)map.get("custom");
            if (custom != null) {
                totalPlaytimeTicks += StatsParser.getLong(custom, "play_time");
                for (String string : distanceKeys) {
                    distanceTotals.merge(string, StatsParser.getLong(custom, string), Long::sum);
                }
                totalKills += StatsParser.getLong(custom, "mob_kills");
                totalPlayerKills += StatsParser.getLong(custom, "player_kills");
                totalDeaths += StatsParser.getLong(custom, "deaths");
                totalDamageDealt += StatsParser.getLong(custom, "damage_dealt");
                totalDamageTaken += StatsParser.getLong(custom, "damage_taken");
                for (String string : interactionKeys) {
                    interactionTotals.merge(string, StatsParser.getLong(custom, string), Long::sum);
                }
            }
            if ((map3 = (Map)map.get("killed")) != null) {
                for (Map.Entry entry2 : map3.entrySet()) {
                    killedAggregated.merge((String)entry2.getKey(), ((Number)entry2.getValue()).longValue(), Long::sum);
                }
            }
            if ((mined = (Map)map.get("mined")) != null) {
                for (Map.Entry entry : mined.entrySet()) {
                    minedAggregated.merge((String)entry.getKey(), ((Number)entry.getValue()).longValue(), Long::sum);
                }
            }
            if ((crafted = (Map)map.get("crafted")) != null) {
                for (Object entry4 : crafted.entrySet()) {
                    craftedAggregated.merge((String)entry4.getKey(), ((Number)entry4.getValue()).longValue(), Long::sum);
                }
            }
            if ((map2 = (Map)map.get("used")) != null) {
                Object entry4;
                entry4 = map2.entrySet().iterator();
                while (entry4.hasNext()) {
                    Map.Entry entry = (Map.Entry)entry4.next();
                    usedAggregated.merge((String)entry.getKey(), ((Number)entry.getValue()).longValue(), Long::sum);
                }
            }
            if ((broken = (Map)map.get("broken")) == null) continue;
            for (Map.Entry entry6 : broken.entrySet()) {
                brokenAggregated.merge((String)entry6.getKey(), ((Number)entry6.getValue()).longValue(), Long::sum);
            }
        }
        LinkedHashMap<String, Object> playtime = new LinkedHashMap<String, Object>();
        double d = (double)Math.round((double)totalPlaytimeTicks / 72000.0 * 10.0) / 10.0;
        playtime.put("total_hours", d);
        playtime.put("total_formatted", StatsUtils.formatPlaytime(totalPlaytimeTicks));
        result.put("playtime", playtime);
        result.put("distance", StatsParser.buildAggregatedDistanceSection(distanceTotals));
        LinkedHashMap<String, Object> linkedHashMap = new LinkedHashMap<String, Object>();
        linkedHashMap.put("total_kills", totalKills);
        linkedHashMap.put("total_deaths", totalDeaths);
        linkedHashMap.put("total_player_kills", totalPlayerKills);
        linkedHashMap.put("damage_dealt_hearts", (double)Math.round((double)totalDamageDealt / 20.0 * 10.0) / 10.0);
        linkedHashMap.put("damage_taken_hearts", (double)Math.round((double)totalDamageTaken / 20.0 * 10.0) / 10.0);
        linkedHashMap.put("unique_mobs_killed", killedAggregated.size());
        linkedHashMap.put("top_mobs_killed", StatsParser.extractTopNFromLongMap(killedAggregated, "mob", 5));
        result.put("combat", linkedHashMap);
        LinkedHashMap<String, Object> mining = new LinkedHashMap<String, Object>();
        long totalBlocksMined = minedAggregated.values().stream().mapToLong(Long::longValue).sum();
        mining.put("total_blocks", totalBlocksMined);
        mining.put("top_blocks", StatsParser.extractTopNFromLongMap(minedAggregated, "block", 5));
        result.put("mining", mining);
        LinkedHashMap<String, Object> crafting = new LinkedHashMap<String, Object>();
        long l = craftedAggregated.values().stream().mapToLong(Long::longValue).sum();
        crafting.put("total_items", l);
        crafting.put("top_items", StatsParser.extractTopNFromLongMap(craftedAggregated, "item", 5));
        result.put("items_crafted", crafting);
        LinkedHashMap<String, Object> itemsUsed = new LinkedHashMap<String, Object>();
        long totalItemsUsed = usedAggregated.values().stream().mapToLong(Long::longValue).sum();
        itemsUsed.put("total", totalItemsUsed);
        itemsUsed.put("top_items", StatsParser.extractTopNFromLongMap(usedAggregated, "item", 5));
        result.put("items_used", itemsUsed);
        LinkedHashMap<String, Object> toolsBroken = new LinkedHashMap<String, Object>();
        long totalToolsBroken = brokenAggregated.values().stream().mapToLong(Long::longValue).sum();
        toolsBroken.put("total", totalToolsBroken);
        toolsBroken.put("top_items", StatsParser.extractTopNFromLongMap(brokenAggregated, "item", 5));
        result.put("tools_broken", toolsBroken);
        LinkedHashMap interactions = new LinkedHashMap();
        interactions.put("chests_opened", interactionTotals.get("open_chest"));
        interactions.put("furnaces_used", interactionTotals.get("interact_with_furnace"));
        interactions.put("crafting_tables_used", interactionTotals.get("interact_with_crafting_table"));
        interactions.put("beds_slept_in", interactionTotals.get("sleep_in_bed"));
        interactions.put("villagers_talked_to", interactionTotals.get("talked_to_villager"));
        interactions.put("containers_looted", interactionTotals.get("looted_stat"));
        result.put("interactions", interactions);
        return result;
    }

    private static Map<String, Object> buildAggregatedDistanceSection(Map<String, Long> distanceTotals) {
        LinkedHashMap<String, Object> distance = new LinkedHashMap<String, Object>();
        String[][] mappings = new String[][]{{"walk_one_cm", "walking_km"}, {"sprint_one_cm", "sprinting_km"}, {"swim_one_cm", "swimming_km"}, {"fly_one_cm", "flying_km"}, {"fall_one_cm", "falling_km"}, {"climb_one_cm", "climbing_km"}, {"crouch_one_cm", "crouching_km"}, {"walk_on_water_one_cm", "walking_on_water_km"}, {"walk_under_water_one_cm", "walking_underwater_km"}, {"boat_one_cm", "boating_km"}, {"minecart_one_cm", "minecart_km"}, {"horse_one_cm", "horse_km"}, {"pig_one_cm", "pig_km"}, {"strider_one_cm", "strider_km"}, {"aviate_one_cm", "elytra_km"}};
        long totalCm = distanceTotals.values().stream().mapToLong(Long::longValue).sum();
        distance.put("total_km", StatsParser.cmToKm(totalCm));
        for (String[] mapping : mappings) {
            String rawKey = mapping[0];
            String outputKey = mapping[1];
            long cm = distanceTotals.getOrDefault(rawKey, 0L);
            if (cm <= 0L) continue;
            distance.put(outputKey, StatsParser.cmToKm(cm));
        }
        return distance;
    }

    private static List<Map<String, Object>> extractTopNFromLongMap(Map<String, Long> source, String keyName, int limit) {
        return source.entrySet().stream().sorted((a, b) -> Long.compare((Long)b.getValue(), (Long)a.getValue())).limit(limit).map(entry -> {
            LinkedHashMap<String, Object> item = new LinkedHashMap<String, Object>();
            item.put(keyName, entry.getKey());
            item.put("count", entry.getValue());
            return item;
        }).collect(Collectors.toList());
    }
}

