/*
 * Decompiled with CFR 0.152.
 */
package com.crittafur.statsapi;

import com.crittafur.statsapi.AdvancementMetadataProvider;
import com.crittafur.statsapi.AdvancementsReader;
import com.crittafur.statsapi.Config;
import com.crittafur.statsapi.PlayerTracker;
import com.crittafur.statsapi.StatsReader;
import com.crittafur.statsapi.StatsUtils;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.sun.net.httpserver.HttpExchange;
import com.sun.net.httpserver.HttpHandler;
import com.sun.net.httpserver.HttpServer;
import java.io.IOException;
import java.io.OutputStream;
import java.io.Serializable;
import java.net.InetSocketAddress;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.level.ServerPlayer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StatsHttpServer {
    private static final Logger LOGGER = LoggerFactory.getLogger((String)"StatsAPI");
    private static final int CACHE_MAX_SIZE = 100;
    private static final int SERVER_THREAD_TIMEOUT_SECONDS = 5;
    private static final String UUID_PATTERN = "[a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12}";
    private static final Set<String> VALID_STATS = Set.of("play_time", "deaths", "mob_kills", "kills", "player_kills", "damage_dealt", "damage_taken", "blocks_mined", "total_blocks_mined", "items_crafted", "items_used", "distance", "total_distance", "walking_distance", "sprinting_distance", "swimming_distance", "flying_distance", "elytra_distance", "horse_distance", "boating_distance", "chests_opened", "beds_slept_in", "containers_looted", "tools_broken", "advancements_completed", "jumps");
    private final int port;
    private final MinecraftServer minecraftServer;
    private final StatsReader statsReader;
    private final AdvancementsReader advancementsReader;
    private final PlayerTracker playerTracker;
    private final AdvancementMetadataProvider advancementMetadataProvider;
    private final Gson gson;
    private HttpServer server;
    private ThreadPoolExecutor executor;
    private long startTimeMillis;
    private final String bindAddress;
    private final int threadPoolSize;
    private final int shutdownTimeoutSeconds;
    private final String corsAllowedOrigins;
    private final int maxLeaderboardLimit;
    private final boolean cacheEnabled;
    private final Cache<String, CachedResponse> healthCache;
    private final Cache<String, CachedResponse> playersCache;
    private final Cache<String, CachedResponse> statsCache;
    private final Cache<String, CachedResponse> serverStatsCache;
    private final Cache<String, CachedResponse> leaderboardCache;
    private static final ThreadLocal<RequestContext> REQUEST_CONTEXT = new ThreadLocal();

    public StatsHttpServer(int port, String bindAddress, int threadPoolSize, int shutdownTimeoutSeconds, String corsAllowedOrigins, int maxLeaderboardLimit, boolean cacheEnabled, int realtimeCacheTtlSeconds, int statsCacheTtlSeconds, MinecraftServer minecraftServer, PlayerTracker playerTracker, AdvancementMetadataProvider advancementMetadataProvider) {
        this.port = port;
        this.bindAddress = bindAddress;
        this.threadPoolSize = threadPoolSize;
        this.shutdownTimeoutSeconds = shutdownTimeoutSeconds;
        this.corsAllowedOrigins = corsAllowedOrigins;
        this.maxLeaderboardLimit = maxLeaderboardLimit;
        this.cacheEnabled = cacheEnabled;
        this.minecraftServer = minecraftServer;
        this.advancementsReader = new AdvancementsReader(minecraftServer);
        this.statsReader = new StatsReader(minecraftServer, this.advancementsReader);
        this.playerTracker = playerTracker;
        this.advancementMetadataProvider = advancementMetadataProvider;
        this.gson = new GsonBuilder().setPrettyPrinting().serializeNulls().create();
        if (cacheEnabled) {
            this.healthCache = this.buildCache(realtimeCacheTtlSeconds);
            this.playersCache = this.buildCache(realtimeCacheTtlSeconds);
            this.statsCache = this.buildCache(statsCacheTtlSeconds);
            this.serverStatsCache = this.buildCache(statsCacheTtlSeconds);
            this.leaderboardCache = this.buildCache(statsCacheTtlSeconds);
            LOGGER.info("[HTTP] Response caching enabled (realtime: {}s, stats: {}s)", (Object)realtimeCacheTtlSeconds, (Object)statsCacheTtlSeconds);
        } else {
            this.healthCache = null;
            this.playersCache = null;
            this.statsCache = null;
            this.serverStatsCache = null;
            this.leaderboardCache = null;
            LOGGER.info("[HTTP] Response caching disabled");
        }
    }

    private Cache<String, CachedResponse> buildCache(int ttlSeconds) {
        return CacheBuilder.newBuilder().expireAfterWrite((long)ttlSeconds, TimeUnit.SECONDS).maximumSize(100L).build();
    }

    private Map<String, Object> wrapUnmodifiable(Map<String, Object> original) {
        return Collections.unmodifiableMap(original);
    }

    private static Map<String, String> parseQuery(String query) {
        HashMap<String, String> params = new HashMap<String, String>();
        if (query == null || query.isEmpty()) {
            return params;
        }
        if (query.length() > 1024) {
            return params;
        }
        String[] parts = query.split("&");
        int maxParams = Math.min(parts.length, 10);
        for (int i = 0; i < maxParams; ++i) {
            String[] pair;
            String param = parts[i];
            if (param.length() > 256 || (pair = param.split("=", 2)).length != 2) continue;
            try {
                String key = URLDecoder.decode(pair[0], StandardCharsets.UTF_8);
                String value = URLDecoder.decode(pair[1], StandardCharsets.UTF_8);
                params.put(key, value);
                continue;
            }
            catch (IllegalArgumentException e) {
                LOGGER.debug("[HTTP] Skipping malformed query parameter: {}", (Object)param);
            }
        }
        return params;
    }

    public void start() throws IOException {
        this.startTimeMillis = System.currentTimeMillis();
        this.server = HttpServer.create(new InetSocketAddress(this.bindAddress, this.port), 0);
        this.server.createContext("/", this.wrapWithLogging(new RootHandler(), "/"));
        this.server.createContext("/health", this.wrapWithLogging(new HealthHandler(), "/health"));
        this.server.createContext("/players", this.wrapWithLogging(new PlayersHandler(), "/players"));
        this.server.createContext("/stats", this.wrapWithLogging(new StatsHandler(), "/stats"));
        this.server.createContext("/leaderboard", this.wrapWithLogging(new LeaderboardHandler(), "/leaderboard"));
        if (((Boolean)Config.SERVER.debugEndpointEnabled.get()).booleanValue()) {
            this.server.createContext("/debug", this.wrapWithLogging(new DebugHandler(), "/debug"));
            LOGGER.info("[HTTP] Debug endpoint enabled at /debug");
        }
        this.executor = (ThreadPoolExecutor)Executors.newFixedThreadPool(this.threadPoolSize);
        this.server.setExecutor(this.executor);
        this.server.start();
    }

    public void stop() {
        if (this.healthCache != null) {
            this.healthCache.invalidateAll();
        }
        if (this.playersCache != null) {
            this.playersCache.invalidateAll();
        }
        if (this.statsCache != null) {
            this.statsCache.invalidateAll();
        }
        if (this.serverStatsCache != null) {
            this.serverStatsCache.invalidateAll();
        }
        if (this.leaderboardCache != null) {
            this.leaderboardCache.invalidateAll();
        }
        if (this.server != null) {
            this.server.stop(this.shutdownTimeoutSeconds);
        }
        if (this.executor != null) {
            this.executor.shutdown();
            try {
                if (!this.executor.awaitTermination(this.shutdownTimeoutSeconds, TimeUnit.SECONDS)) {
                    this.executor.shutdownNow();
                    LOGGER.warn("[HTTP] Executor did not terminate gracefully");
                }
            }
            catch (InterruptedException e) {
                this.executor.shutdownNow();
                Thread.currentThread().interrupt();
            }
        }
    }

    private void sendJsonResponse(HttpExchange exchange, int statusCode, Object data) throws IOException {
        String json = this.gson.toJson(data);
        byte[] bytes = json.getBytes(StandardCharsets.UTF_8);
        exchange.getResponseHeaders().set("Content-Type", "application/json");
        exchange.getResponseHeaders().set("Access-Control-Allow-Origin", this.corsAllowedOrigins);
        exchange.getResponseHeaders().set("Access-Control-Allow-Methods", "GET, OPTIONS");
        exchange.sendResponseHeaders(statusCode, bytes.length);
        try (OutputStream os = exchange.getResponseBody();){
            os.write(bytes);
        }
    }

    private boolean handleCors(HttpExchange exchange) throws IOException {
        if ("OPTIONS".equals(exchange.getRequestMethod())) {
            exchange.getResponseHeaders().set("Access-Control-Allow-Origin", this.corsAllowedOrigins);
            exchange.getResponseHeaders().set("Access-Control-Allow-Methods", "GET, OPTIONS");
            exchange.getResponseHeaders().set("Access-Control-Allow-Headers", "Content-Type");
            exchange.sendResponseHeaders(204, -1L);
            return true;
        }
        return false;
    }

    private void markCacheHit() {
        RequestContext ctx = REQUEST_CONTEXT.get();
        if (ctx != null) {
            ctx.cacheHit = true;
        }
    }

    private HttpHandler wrapWithLogging(HttpHandler handler, String basePath) {
        return exchange -> {
            String fullPath;
            String method;
            long durationMs;
            if (!((Boolean)Config.SERVER.requestLogging.get()).booleanValue()) {
                handler.handle(exchange);
                return;
            }
            long startNanos = System.nanoTime();
            RequestContext ctx = new RequestContext();
            REQUEST_CONTEXT.set(ctx);
            try {
                handler.handle(exchange);
                REQUEST_CONTEXT.remove();
                durationMs = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startNanos);
                method = exchange.getRequestMethod();
                String path = exchange.getRequestURI().getPath();
                String query = exchange.getRequestURI().getQuery();
                fullPath = query != null ? path + "?" + query : path;
            }
            catch (Throwable throwable) {
                REQUEST_CONTEXT.remove();
                long durationMs2 = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startNanos);
                String method2 = exchange.getRequestMethod();
                String path = exchange.getRequestURI().getPath();
                String query = exchange.getRequestURI().getQuery();
                String fullPath2 = query != null ? path + "?" + query : path;
                int responseCode = exchange.getResponseCode();
                String cacheInfo = "";
                if (this.cacheEnabled && ctx.cacheHit) {
                    cacheInfo = " [cache hit]";
                } else if (this.cacheEnabled) {
                    cacheInfo = " [cache miss]";
                }
                LOGGER.info("[HTTP] {} {} -> {} ({}ms){}", new Object[]{method2, fullPath2, responseCode, durationMs2, cacheInfo});
                throw throwable;
            }
            int responseCode = exchange.getResponseCode();
            String cacheInfo = "";
            if (this.cacheEnabled && ctx.cacheHit) {
                cacheInfo = " [cache hit]";
            } else if (this.cacheEnabled) {
                cacheInfo = " [cache miss]";
            }
            LOGGER.info("[HTTP] {} {} -> {} ({}ms){}", new Object[]{method, fullPath, responseCode, durationMs, cacheInfo});
        };
    }

    private <T> T executeOnServerThread(Supplier<T> task) {
        if (this.minecraftServer.isSameThread()) {
            return task.get();
        }
        CompletableFuture future = new CompletableFuture();
        this.minecraftServer.execute(() -> {
            try {
                future.complete(task.get());
            }
            catch (Exception e) {
                future.completeExceptionally(e);
            }
        });
        try {
            return future.get(5L, TimeUnit.SECONDS);
        }
        catch (Exception e) {
            throw new RuntimeException("Failed to execute on server thread", e);
        }
    }

    private void enrichAdvancementsWithMetadata(Map<String, Object> advancements) {
        Object categoriesObj = advancements.get("categories");
        if (!(categoriesObj instanceof Map)) {
            return;
        }
        Map categories = (Map)categoriesObj;
        for (Object categoryObj : categories.values()) {
            Map category;
            Object listObj;
            if (!(categoryObj instanceof Map) || !((listObj = (category = (Map)categoryObj).get("list")) instanceof List)) continue;
            List list = (List)listObj;
            for (Map entry : list) {
                String id = (String)entry.get("id");
                if (id == null) continue;
                AdvancementMetadataProvider.AdvancementMetadata metadata = this.advancementMetadataProvider.getMetadata(id);
                if (metadata != null) {
                    entry.put("title", metadata.title());
                    entry.put("description", metadata.description());
                    entry.put("icon", metadata.icon());
                    entry.put("frame", metadata.frame());
                    continue;
                }
                entry.put("title", null);
                entry.put("description", null);
                entry.put("icon", null);
                entry.put("frame", null);
            }
        }
    }

    private class RootHandler
    implements HttpHandler {
        private RootHandler() {
        }

        @Override
        public void handle(HttpExchange exchange) throws IOException {
            if (StatsHttpServer.this.handleCors(exchange)) {
                return;
            }
            String path = exchange.getRequestURI().getPath();
            if (!path.equals("/")) {
                StatsHttpServer.this.sendJsonResponse(exchange, 404, Map.of("error", "Not found"));
                return;
            }
            try {
                LinkedHashMap<String, Object> response = new LinkedHashMap<String, Object>();
                response.put("name", "Stats API");
                response.put("version", "1.0.0");
                response.put("description", "REST API for Minecraft server statistics");
                ArrayList<Map<String, Object>> endpoints = new ArrayList<Map<String, Object>>();
                endpoints.add(this.createEndpoint("GET", "/", "API documentation and endpoint discovery"));
                endpoints.add(this.createEndpoint("GET", "/health", "Server health check and basic info"));
                Map<String, Object> players = this.createEndpoint("GET", "/players", "List all tracked players with positions and session info");
                LinkedHashMap<String, Map<String, List<String>>> playersParams = new LinkedHashMap<String, Map<String, List<String>>>();
                playersParams.put("filter", Map.of("type", "string", "required", false, "description", "Filter by player status", "valid_values", List.of("online", "offline")));
                players.put("parameters", playersParams);
                endpoints.add(players);
                endpoints.add(this.createEndpoint("GET", "/stats", "Get statistics for all players"));
                endpoints.add(this.createEndpoint("GET", "/stats/server", "Get aggregated server-wide statistics"));
                Map<String, Object> singleStats = this.createEndpoint("GET", "/stats/{uuid}", "Get statistics for a specific player");
                LinkedHashMap<String, Map<String, String>> singleStatsParams = new LinkedHashMap<String, Map<String, String>>();
                singleStatsParams.put("uuid", Map.of("type", "string", "required", true, "description", "Player UUID (with or without dashes)"));
                singleStatsParams.put("include_ranks", Map.of("type", "boolean", "default", false, "description", "Include player's rank for each stat (e.g., \"Rank 3 of 15\")"));
                singleStatsParams.put("include_advancements", Map.of("type", "boolean", "default", true, "description", "Include player advancements (filtered, grouped by category)"));
                singleStatsParams.put("include_partial", Map.of("type", "boolean", "default", false, "description", "Include incomplete advancements with criteria progress"));
                singleStats.put("parameters", singleStatsParams);
                endpoints.add(singleStats);
                Map<String, Object> leaderboard = this.createEndpoint("GET", "/leaderboard", "Get ranked leaderboard for a statistic");
                LinkedHashMap<String, Map<String, Object>> leaderboardParams = new LinkedHashMap<String, Map<String, Object>>();
                leaderboardParams.put("stat", Map.of("type", "string", "default", "play_time", "description", "Statistic to rank by", "valid_values", VALID_STATS));
                leaderboardParams.put("limit", Map.of("type", "integer", "default", 10, "max", StatsHttpServer.this.maxLeaderboardLimit, "description", "Number of entries to return"));
                leaderboard.put("parameters", leaderboardParams);
                endpoints.add(leaderboard);
                if (((Boolean)Config.SERVER.debugEndpointEnabled.get()).booleanValue()) {
                    endpoints.add(this.createEndpoint("GET", "/debug", "Server diagnostics and debugging information"));
                }
                response.put("endpoints", endpoints);
                response.put("timestamp", System.currentTimeMillis());
                StatsHttpServer.this.sendJsonResponse(exchange, 200, response);
            }
            catch (Exception e) {
                LOGGER.error("[HTTP] Error handling / request", (Throwable)e);
                StatsHttpServer.this.sendJsonResponse(exchange, 500, Map.of("error", "Internal server error"));
            }
        }

        private Map<String, Object> createEndpoint(String method, String path, String description) {
            LinkedHashMap<String, Object> endpoint = new LinkedHashMap<String, Object>();
            endpoint.put("method", method);
            endpoint.put("path", path);
            endpoint.put("description", description);
            return endpoint;
        }
    }

    private class HealthHandler
    implements HttpHandler {
        private static final String CACHE_KEY = "health";

        private HealthHandler() {
        }

        @Override
        public void handle(HttpExchange exchange) throws IOException {
            if (StatsHttpServer.this.handleCors(exchange)) {
                return;
            }
            try {
                CachedResponse cached;
                if (StatsHttpServer.this.cacheEnabled && (cached = (CachedResponse)StatsHttpServer.this.healthCache.getIfPresent((Object)CACHE_KEY)) != null) {
                    StatsHttpServer.this.markCacheHit();
                    StatsHttpServer.this.sendJsonResponse(exchange, 200, cached.data());
                    return;
                }
                Map response = StatsHttpServer.this.executeOnServerThread(() -> {
                    LinkedHashMap<String, Object> result = new LinkedHashMap<String, Object>();
                    result.put("status", "ok");
                    result.put("serverName", StatsHttpServer.this.minecraftServer.getMotd());
                    result.put("version", StatsHttpServer.this.minecraftServer.getServerVersion());
                    result.put("timestamp", System.currentTimeMillis());
                    return result;
                });
                if (StatsHttpServer.this.cacheEnabled) {
                    StatsHttpServer.this.healthCache.put((Object)CACHE_KEY, (Object)new CachedResponse(StatsHttpServer.this.wrapUnmodifiable(response), System.currentTimeMillis()));
                }
                StatsHttpServer.this.sendJsonResponse(exchange, 200, response);
            }
            catch (Exception e) {
                LOGGER.error("[HTTP] Error handling /health request", (Throwable)e);
                StatsHttpServer.this.sendJsonResponse(exchange, 500, Map.of("error", "Internal server error"));
            }
        }
    }

    private class PlayersHandler
    implements HttpHandler {
        private static final String CACHE_KEY_ALL = "players:all";
        private static final String CACHE_KEY_ONLINE = "players:online";
        private static final String CACHE_KEY_OFFLINE = "players:offline";

        private PlayersHandler() {
        }

        @Override
        public void handle(HttpExchange exchange) throws IOException {
            if (StatsHttpServer.this.handleCors(exchange)) {
                return;
            }
            try {
                CachedResponse cached;
                String cacheKey;
                String query = exchange.getRequestURI().getQuery();
                Map<String, String> params = StatsHttpServer.parseQuery(query);
                String filter = params.get("filter");
                if (filter != null && !filter.equals("online") && !filter.equals("offline")) {
                    StatsHttpServer.this.sendJsonResponse(exchange, 400, Map.of("error", "Invalid filter parameter", "valid_values", List.of("online", "offline")));
                    return;
                }
                String string = filter == null ? CACHE_KEY_ALL : (cacheKey = filter.equals("online") ? CACHE_KEY_ONLINE : CACHE_KEY_OFFLINE);
                if (StatsHttpServer.this.cacheEnabled && (cached = (CachedResponse)StatsHttpServer.this.playersCache.getIfPresent((Object)cacheKey)) != null) {
                    StatsHttpServer.this.markCacheHit();
                    StatsHttpServer.this.sendJsonResponse(exchange, 200, cached.data());
                    return;
                }
                String finalFilter = filter;
                Map response = StatsHttpServer.this.executeOnServerThread(() -> {
                    Map<String, Map<String, Object>> allSessions = StatsHttpServer.this.playerTracker.getAllSessionInfo();
                    HashMap<String, ServerPlayer> onlinePlayers = new HashMap<String, ServerPlayer>();
                    for (ServerPlayer player : StatsHttpServer.this.minecraftServer.getPlayerList().getPlayers()) {
                        onlinePlayers.put(player.getUUID().toString(), player);
                    }
                    ArrayList playerList = new ArrayList();
                    int onlineCount = 0;
                    for (Map.Entry<String, Map<String, Object>> entry : allSessions.entrySet()) {
                        String uuid = entry.getKey();
                        Map<String, Object> sessionInfo = entry.getValue();
                        boolean isOnline = Boolean.TRUE.equals(sessionInfo.get("online"));
                        if (isOnline) {
                            ++onlineCount;
                        }
                        if (finalFilter != null && (finalFilter.equals("online") && !isOnline || finalFilter.equals("offline") && isOnline)) continue;
                        LinkedHashMap<String, Object> p = new LinkedHashMap<String, Object>();
                        if (isOnline) {
                            ServerPlayer player = (ServerPlayer)onlinePlayers.get(uuid);
                            if (player == null) continue;
                            p.put("name", player.getName().getString());
                            p.put("uuid", uuid);
                            p.putAll(sessionInfo);
                            p.put("position", Map.of("x", Math.round(player.getX()), "y", Math.round(player.getY()), "z", Math.round(player.getZ()), "dimension", player.level().dimension().location().toString()));
                        } else {
                            String name = StatsHttpServer.this.statsReader.getPlayerName(uuid);
                            p.put("name", name);
                            p.put("uuid", uuid);
                            p.putAll(sessionInfo);
                        }
                        playerList.add(p);
                    }
                    LinkedHashMap<String, Serializable> result = new LinkedHashMap<String, Serializable>();
                    result.put("total", Integer.valueOf(playerList.size()));
                    result.put("online", Integer.valueOf(onlineCount));
                    result.put("max", Integer.valueOf(StatsHttpServer.this.minecraftServer.getMaxPlayers()));
                    result.put("players", playerList);
                    result.put("timestamp", Long.valueOf(System.currentTimeMillis()));
                    return result;
                });
                if (StatsHttpServer.this.cacheEnabled) {
                    StatsHttpServer.this.playersCache.put((Object)cacheKey, (Object)new CachedResponse(StatsHttpServer.this.wrapUnmodifiable(response), System.currentTimeMillis()));
                }
                StatsHttpServer.this.sendJsonResponse(exchange, 200, response);
            }
            catch (Exception e) {
                LOGGER.error("[HTTP] Error handling /players request", (Throwable)e);
                StatsHttpServer.this.sendJsonResponse(exchange, 500, Map.of("error", "Internal server error"));
            }
        }
    }

    private class StatsHandler
    implements HttpHandler {
        private static final String ALL_STATS_CACHE_KEY = "all";
        private static final String SERVER_STATS_CACHE_KEY = "server";
        private static final Set<String> EXCLUDED_FROM_ALL_PLAYERS = Set.of("custom", "mined", "killed", "killed_by", "used", "crafted", "picked_up", "dropped", "broken");

        private StatsHandler() {
        }

        @Override
        public void handle(HttpExchange exchange) throws IOException {
            if (StatsHttpServer.this.handleCors(exchange)) {
                return;
            }
            try {
                String path = exchange.getRequestURI().getPath().toLowerCase();
                if (path.equals("/stats/server") || path.equals("/stats/server/")) {
                    this.handleServerStats(exchange);
                } else if (path.matches("/stats/[a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12}")) {
                    String uuid = path.substring("/stats/".length());
                    this.handleSinglePlayer(exchange, uuid);
                } else if (path.equals("/stats") || path.equals("/stats/")) {
                    this.handleAllPlayers(exchange);
                } else {
                    StatsHttpServer.this.sendJsonResponse(exchange, 404, Map.of("error", "Not found"));
                }
            }
            catch (Exception e) {
                LOGGER.error("[HTTP] Error handling /stats request", (Throwable)e);
                StatsHttpServer.this.sendJsonResponse(exchange, 500, Map.of("error", "Internal server error"));
            }
        }

        private void handleServerStats(HttpExchange exchange) throws IOException {
            CachedResponse cached;
            if (StatsHttpServer.this.cacheEnabled && (cached = (CachedResponse)StatsHttpServer.this.serverStatsCache.getIfPresent((Object)SERVER_STATS_CACHE_KEY)) != null) {
                StatsHttpServer.this.markCacheHit();
                StatsHttpServer.this.sendJsonResponse(exchange, 200, cached.data());
                return;
            }
            Map response = StatsHttpServer.this.executeOnServerThread(() -> {
                Map<String, Map<String, Object>> allStats = StatsHttpServer.this.statsReader.getAllPlayerStats();
                Map<String, Object> serverStats = StatsHttpServer.this.statsReader.getServerStats(allStats);
                serverStats.put("timestamp", System.currentTimeMillis());
                long totalCompleted = 0L;
                HashSet<String> allUniqueIds = new HashSet<String>();
                HashSet<String> vanillaUniqueIds = new HashSet<String>();
                HashSet<String> moddedUniqueIds = new HashSet<String>();
                long vanillaTotalCompleted = 0L;
                long moddedTotalCompleted = 0L;
                for (String uuid : allStats.keySet()) {
                    Set<String> completedIds = StatsHttpServer.this.advancementsReader.getCompletedAdvancementIds(uuid);
                    totalCompleted += (long)completedIds.size();
                    allUniqueIds.addAll(completedIds);
                    for (String id : completedIds) {
                        if (id.startsWith("minecraft:")) {
                            vanillaUniqueIds.add(id);
                            ++vanillaTotalCompleted;
                            continue;
                        }
                        moddedUniqueIds.add(id);
                        ++moddedTotalCompleted;
                    }
                }
                LinkedHashMap<String, Serializable> advancements = new LinkedHashMap<String, Serializable>();
                advancements.put("total_completed", Long.valueOf(totalCompleted));
                advancements.put("unique_completed", Integer.valueOf(allUniqueIds.size()));
                LinkedHashMap<String, Number> vanilla = new LinkedHashMap<String, Number>();
                vanilla.put("total_completed", vanillaTotalCompleted);
                vanilla.put("unique_completed", vanillaUniqueIds.size());
                advancements.put("vanilla", vanilla);
                LinkedHashMap<String, Number> modded = new LinkedHashMap<String, Number>();
                modded.put("total_completed", moddedTotalCompleted);
                modded.put("unique_completed", moddedUniqueIds.size());
                advancements.put("modded", modded);
                serverStats.put("advancements", advancements);
                return serverStats;
            });
            if (StatsHttpServer.this.cacheEnabled) {
                StatsHttpServer.this.serverStatsCache.put((Object)SERVER_STATS_CACHE_KEY, (Object)new CachedResponse(StatsHttpServer.this.wrapUnmodifiable(response), System.currentTimeMillis()));
            }
            StatsHttpServer.this.sendJsonResponse(exchange, 200, response);
        }

        private void handleAllPlayers(HttpExchange exchange) throws IOException {
            CachedResponse cached;
            if (StatsHttpServer.this.cacheEnabled && (cached = (CachedResponse)StatsHttpServer.this.statsCache.getIfPresent((Object)ALL_STATS_CACHE_KEY)) != null) {
                StatsHttpServer.this.markCacheHit();
                StatsHttpServer.this.sendJsonResponse(exchange, 200, cached.data());
                return;
            }
            Map response = StatsHttpServer.this.executeOnServerThread(() -> {
                Map<String, Map<String, Object>> allStats = StatsHttpServer.this.statsReader.getAllPlayerStats();
                LinkedHashMap filteredStats = new LinkedHashMap();
                for (Map.Entry<String, Map<String, Object>> entry : allStats.entrySet()) {
                    String uuid = entry.getKey();
                    Map<String, Object> stats = entry.getValue();
                    LinkedHashMap<String, Object> filtered = new LinkedHashMap<String, Object>();
                    for (Map.Entry<String, Object> statEntry : stats.entrySet()) {
                        if (EXCLUDED_FROM_ALL_PLAYERS.contains(statEntry.getKey())) continue;
                        filtered.put(statEntry.getKey(), statEntry.getValue());
                    }
                    Map<String, Object> sessionInfo = StatsHttpServer.this.playerTracker.getSessionInfo(uuid);
                    filtered.putAll(sessionInfo);
                    Map<String, Object> playerAdvancements = StatsHttpServer.this.advancementsReader.getPlayerAdvancements(uuid, false);
                    if (playerAdvancements != null) {
                        filtered.put("advancements_completed", playerAdvancements.get("completed"));
                    } else {
                        filtered.put("advancements_completed", 0);
                    }
                    filteredStats.put(uuid, filtered);
                }
                LinkedHashMap<String, Serializable> result = new LinkedHashMap<String, Serializable>();
                result.put("playerCount", Integer.valueOf(filteredStats.size()));
                result.put("timestamp", Long.valueOf(System.currentTimeMillis()));
                result.put("players", filteredStats);
                return result;
            });
            if (StatsHttpServer.this.cacheEnabled) {
                StatsHttpServer.this.statsCache.put((Object)ALL_STATS_CACHE_KEY, (Object)new CachedResponse(StatsHttpServer.this.wrapUnmodifiable(response), System.currentTimeMillis()));
            }
            StatsHttpServer.this.sendJsonResponse(exchange, 200, response);
        }

        private void handleSinglePlayer(HttpExchange exchange, String uuid) throws IOException {
            CachedResponse cached;
            String query = exchange.getRequestURI().getQuery();
            Map<String, String> params = StatsHttpServer.parseQuery(query);
            boolean includeRanks = "true".equalsIgnoreCase(params.get("include_ranks"));
            boolean includeAdvancements = !"false".equalsIgnoreCase(params.get("include_advancements"));
            boolean includePartial = "true".equalsIgnoreCase(params.get("include_partial"));
            String cacheKey = uuid.toLowerCase() + (includeRanks ? ":ranks" : "") + (includeAdvancements ? "" : ":noadv") + (includePartial ? ":partial" : "");
            if (StatsHttpServer.this.cacheEnabled && (cached = (CachedResponse)StatsHttpServer.this.statsCache.getIfPresent((Object)cacheKey)) != null) {
                StatsHttpServer.this.markCacheHit();
                StatsHttpServer.this.sendJsonResponse(exchange, 200, cached.data());
                return;
            }
            boolean finalIncludeRanks = includeRanks;
            boolean finalIncludeAdvancements = includeAdvancements;
            boolean finalIncludePartial = includePartial;
            Map response = StatsHttpServer.this.executeOnServerThread(() -> {
                Map<String, Object> advancements;
                if (finalIncludeRanks) {
                    Map<String, Object> advancements2;
                    String normalizedUuid;
                    Map<String, Map<String, Object>> allStats = StatsHttpServer.this.statsReader.getAllPlayerStats();
                    Map<String, Object> stats = allStats.get(normalizedUuid = StatsUtils.normalizeUuid(uuid));
                    if (stats == null) {
                        return null;
                    }
                    Map<String, Object> sessionInfo = StatsHttpServer.this.playerTracker.getSessionInfo(uuid);
                    stats.putAll(sessionInfo);
                    Map<String, Map<String, Object>> ranks = StatsHttpServer.this.statsReader.getRanksForPlayer(uuid, VALID_STATS, allStats);
                    if (ranks != null) {
                        stats.put("ranks", ranks);
                    }
                    if (finalIncludeAdvancements && (advancements2 = StatsHttpServer.this.advancementsReader.getPlayerAdvancements(uuid, finalIncludePartial)) != null) {
                        StatsHttpServer.this.enrichAdvancementsWithMetadata(advancements2);
                        stats.put("advancements", advancements2);
                    }
                    return stats;
                }
                Map<String, Object> stats = StatsHttpServer.this.statsReader.getPlayerStats(uuid);
                if (stats == null) {
                    return null;
                }
                Map<String, Object> sessionInfo = StatsHttpServer.this.playerTracker.getSessionInfo(uuid);
                stats.putAll(sessionInfo);
                if (finalIncludeAdvancements && (advancements = StatsHttpServer.this.advancementsReader.getPlayerAdvancements(uuid, finalIncludePartial)) != null) {
                    StatsHttpServer.this.enrichAdvancementsWithMetadata(advancements);
                    stats.put("advancements", advancements);
                }
                return stats;
            });
            if (response == null) {
                StatsHttpServer.this.sendJsonResponse(exchange, 404, Map.of("error", "Player not found"));
            } else {
                if (StatsHttpServer.this.cacheEnabled) {
                    StatsHttpServer.this.statsCache.put((Object)cacheKey, (Object)new CachedResponse(StatsHttpServer.this.wrapUnmodifiable(response), System.currentTimeMillis()));
                }
                StatsHttpServer.this.sendJsonResponse(exchange, 200, response);
            }
        }
    }

    private class LeaderboardHandler
    implements HttpHandler {
        private LeaderboardHandler() {
        }

        @Override
        public void handle(HttpExchange exchange) throws IOException {
            if (StatsHttpServer.this.handleCors(exchange)) {
                return;
            }
            try {
                CachedResponse cached;
                int limit;
                String query = exchange.getRequestURI().getQuery();
                Map<String, String> params = StatsHttpServer.parseQuery(query);
                String stat = params.getOrDefault("stat", "play_time");
                if (!VALID_STATS.contains(stat)) {
                    StatsHttpServer.this.sendJsonResponse(exchange, 400, Map.of("error", "Invalid stat parameter", "valid_stats", VALID_STATS));
                    return;
                }
                try {
                    limit = Integer.parseInt(params.getOrDefault("limit", "10"));
                }
                catch (NumberFormatException e) {
                    StatsHttpServer.this.sendJsonResponse(exchange, 400, Map.of("error", "Invalid limit parameter"));
                    return;
                }
                if (limit < 1) {
                    StatsHttpServer.this.sendJsonResponse(exchange, 400, Map.of("error", "Limit must be positive"));
                    return;
                }
                int finalLimit = limit = Math.min(limit, StatsHttpServer.this.maxLeaderboardLimit);
                String finalStat = stat;
                String cacheKey = stat + ":" + limit;
                if (StatsHttpServer.this.cacheEnabled && (cached = (CachedResponse)StatsHttpServer.this.leaderboardCache.getIfPresent((Object)cacheKey)) != null) {
                    StatsHttpServer.this.markCacheHit();
                    StatsHttpServer.this.sendJsonResponse(exchange, 200, cached.data());
                    return;
                }
                Map response = StatsHttpServer.this.executeOnServerThread(() -> {
                    List<Map<String, Object>> leaderboard = StatsHttpServer.this.statsReader.getLeaderboard(finalStat, finalLimit);
                    for (Map<String, Object> entry : leaderboard) {
                        String uuid = (String)entry.get("uuid");
                        Map<String, Object> sessionInfo = StatsHttpServer.this.playerTracker.getSessionInfo(uuid);
                        entry.putAll(sessionInfo);
                    }
                    LinkedHashMap<String, Object> result = new LinkedHashMap<String, Object>();
                    result.put("stat", finalStat);
                    result.put("timestamp", System.currentTimeMillis());
                    result.put("entries", leaderboard);
                    return result;
                });
                if (StatsHttpServer.this.cacheEnabled) {
                    StatsHttpServer.this.leaderboardCache.put((Object)cacheKey, (Object)new CachedResponse(StatsHttpServer.this.wrapUnmodifiable(response), System.currentTimeMillis()));
                }
                StatsHttpServer.this.sendJsonResponse(exchange, 200, response);
            }
            catch (Exception e) {
                LOGGER.error("[HTTP] Error handling /leaderboard request", (Throwable)e);
                StatsHttpServer.this.sendJsonResponse(exchange, 500, Map.of("error", "Internal server error"));
            }
        }
    }

    private class DebugHandler
    implements HttpHandler {
        private static final String MOD_VERSION = "1.2.1";

        private DebugHandler() {
        }

        @Override
        public void handle(HttpExchange exchange) throws IOException {
            if (StatsHttpServer.this.handleCors(exchange)) {
                return;
            }
            if (!((Boolean)Config.SERVER.debugEndpointEnabled.get()).booleanValue()) {
                StatsHttpServer.this.sendJsonResponse(exchange, 404, Map.of("error", "Not found"));
                return;
            }
            try {
                LinkedHashMap<String, Map<Object, Object>> response = new LinkedHashMap<String, Map<Object, Object>>();
                long uptimeSeconds = (System.currentTimeMillis() - StatsHttpServer.this.startTimeMillis) / 1000L;
                LinkedHashMap<String, Object> statsApi = new LinkedHashMap<String, Object>();
                statsApi.put("version", MOD_VERSION);
                statsApi.put("uptime", this.formatUptime(uptimeSeconds));
                statsApi.put("uptimeSeconds", uptimeSeconds);
                response.put("statsApi", statsApi);
                LinkedHashMap<String, Object> config = new LinkedHashMap<String, Object>();
                config.put("httpPort", Config.SERVER.httpPort.get());
                config.put("bindAddress", Config.SERVER.bindAddress.get());
                config.put("threadPoolSize", Config.SERVER.threadPoolSize.get());
                config.put("corsAllowedOrigins", Config.SERVER.corsAllowedOrigins.get());
                config.put("cacheEnabled", Config.SERVER.cacheEnabled.get());
                config.put("realtimeCacheTtlSeconds", Config.SERVER.realtimeCacheTtlSeconds.get());
                config.put("statsCacheTtlSeconds", Config.SERVER.statsCacheTtlSeconds.get());
                config.put("logLevel", Config.SERVER.logLevel.get());
                config.put("requestLogging", Config.SERVER.requestLogging.get());
                response.put("config", config);
                if (StatsHttpServer.this.cacheEnabled) {
                    LinkedHashMap<String, LinkedHashMap<String, Object>> cacheInfo = new LinkedHashMap<String, LinkedHashMap<String, Object>>();
                    LinkedHashMap<String, Object> entries = new LinkedHashMap<String, Object>();
                    this.addCacheEntry(entries, "/health", StatsHttpServer.this.healthCache);
                    this.addCacheEntry(entries, "/players", StatsHttpServer.this.playersCache);
                    this.addCacheEntries(entries, "/stats", StatsHttpServer.this.statsCache);
                    this.addCacheEntries(entries, "/leaderboard", StatsHttpServer.this.leaderboardCache);
                    cacheInfo.put("entries", entries);
                    response.put("cache", cacheInfo);
                } else {
                    response.put("cache", Map.of("enabled", false));
                }
                LinkedHashMap<String, Integer> threadPool = new LinkedHashMap<String, Integer>();
                if (StatsHttpServer.this.executor != null) {
                    threadPool.put("activeThreads", StatsHttpServer.this.executor.getActiveCount());
                    threadPool.put("poolSize", StatsHttpServer.this.executor.getPoolSize());
                    threadPool.put("queuedTasks", StatsHttpServer.this.executor.getQueue().size());
                }
                response.put("threadPool", threadPool);
                LinkedHashMap<String, Integer> players = new LinkedHashMap<String, Integer>();
                players.put("online", StatsHttpServer.this.playerTracker.getOnlineCount());
                players.put("tracked", StatsHttpServer.this.playerTracker.getTrackedCount());
                response.put("players", players);
                StatsHttpServer.this.sendJsonResponse(exchange, 200, response);
            }
            catch (Exception e) {
                LOGGER.error("[HTTP] Error handling /debug request", (Throwable)e);
                StatsHttpServer.this.sendJsonResponse(exchange, 500, Map.of("error", "Internal server error"));
            }
        }

        private String formatUptime(long seconds) {
            long hours = seconds / 3600L;
            long minutes = seconds % 3600L / 60L;
            long secs = seconds % 60L;
            if (hours > 0L) {
                return String.format("%dh %dm %ds", hours, minutes, secs);
            }
            if (minutes > 0L) {
                return String.format("%dm %ds", minutes, secs);
            }
            return String.format("%ds", secs);
        }

        private void addCacheEntry(Map<String, Object> entries, String path, Cache<String, CachedResponse> cache) {
            if (cache == null) {
                return;
            }
            CachedResponse cached = (CachedResponse)cache.getIfPresent((Object)path.substring(1));
            if (cached != null) {
                entries.put(path, this.createCacheEntryInfo(cached));
            }
        }

        private void addCacheEntries(Map<String, Object> entries, String basePath, Cache<String, CachedResponse> cache) {
            if (cache == null) {
                return;
            }
            cache.asMap().forEach((key, cached) -> {
                Object displayPath = basePath;
                if (!key.equals("all") && !key.contains(":")) {
                    displayPath = basePath + "/" + key;
                } else if (key.contains(":")) {
                    displayPath = basePath + "?stat=" + key.replace(":", "&limit=");
                }
                entries.put((String)displayPath, this.createCacheEntryInfo((CachedResponse)cached));
            });
        }

        private Map<String, Object> createCacheEntryInfo(CachedResponse cached) {
            LinkedHashMap<String, Object> info = new LinkedHashMap<String, Object>();
            long ageSeconds = (System.currentTimeMillis() - cached.timestamp()) / 1000L;
            info.put("age", ageSeconds + "s");
            String json = StatsHttpServer.this.gson.toJson(cached.data());
            info.put("size", json.length());
            return info;
        }
    }

    private static class RequestContext {
        boolean cacheHit = false;

        private RequestContext() {
        }
    }

    private record CachedResponse(Map<String, Object> data, long timestamp) {
    }
}

