/*
 * Decompiled with CFR 0.152.
 */
package com.crittafur.statsapi;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.reflect.TypeToken;
import java.io.IOException;
import java.lang.reflect.Type;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.world.entity.player.Player;
import net.minecraft.world.level.storage.LevelResource;
import net.neoforged.bus.api.SubscribeEvent;
import net.neoforged.neoforge.event.entity.player.PlayerEvent;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PlayerTracker {
    private static final Logger LOGGER = LoggerFactory.getLogger((String)"StatsAPI");
    private static final String TRACKER_FILE = "statsapi_sessions.json";
    private final MinecraftServer server;
    private final Path dataFile;
    private final Gson gson;
    private final Map<String, Long> onlineSessions = new ConcurrentHashMap<String, Long>();
    private final Map<String, PlayerData> playerData = new ConcurrentHashMap<String, PlayerData>();
    private final ScheduledExecutorService saveExecutor = Executors.newSingleThreadScheduledExecutor(r -> {
        Thread t = new Thread(r, "StatsAPI-Save");
        t.setDaemon(true);
        return t;
    });
    private final AtomicBoolean savePending = new AtomicBoolean(false);

    public PlayerTracker(MinecraftServer server) {
        this.server = server;
        this.gson = new GsonBuilder().setPrettyPrinting().create();
        this.dataFile = server.getWorldPath(LevelResource.ROOT).resolve(TRACKER_FILE);
        this.loadData();
        for (ServerPlayer player : server.getPlayerList().getPlayers()) {
            this.handleLogin(player);
        }
        LOGGER.info("[Tracker] PlayerTracker initialized with {} known players", (Object)this.playerData.size());
    }

    @SubscribeEvent
    public void onPlayerLogin(PlayerEvent.PlayerLoggedInEvent event) {
        Player player = event.getEntity();
        if (player instanceof ServerPlayer) {
            ServerPlayer player2 = (ServerPlayer)player;
            this.handleLogin(player2);
        }
    }

    @SubscribeEvent
    public void onPlayerLogout(PlayerEvent.PlayerLoggedOutEvent event) {
        Player player = event.getEntity();
        if (player instanceof ServerPlayer) {
            ServerPlayer player2 = (ServerPlayer)player;
            this.handleLogout(player2);
        }
    }

    private void handleLogin(ServerPlayer player) {
        String uuidStr = player.getUUID().toString();
        long now = System.currentTimeMillis();
        String playerName = player.getName().getString();
        this.onlineSessions.put(uuidStr, now);
        this.playerData.compute(uuidStr, (key, existing) -> {
            if (existing == null) {
                return new PlayerData(playerName, now);
            }
            return existing.withLogin(playerName, now);
        });
        LOGGER.debug("[Tracker] Player logged in: {} ({})", (Object)playerName, (Object)uuidStr);
        this.scheduleAsyncSave();
    }

    private void handleLogout(ServerPlayer player) {
        String uuidStr = player.getUUID().toString();
        long now = System.currentTimeMillis();
        double x = player.getX();
        double y = player.getY();
        double z = player.getZ();
        String dimension = player.level().dimension().location().toString();
        Long sessionStart = this.onlineSessions.remove(uuidStr);
        if (sessionStart != null) {
            long duration = now - sessionStart;
            this.playerData.computeIfPresent(uuidStr, (key, existing) -> existing.withLastSeen(now, duration, x, y, z, dimension));
        } else {
            this.playerData.computeIfPresent(uuidStr, (key, existing) -> existing.withLastSeen(now, 0L, x, y, z, dimension));
        }
        LOGGER.debug("[Tracker] Player logged out: {} ({})", (Object)player.getName().getString(), (Object)uuidStr);
        this.scheduleAsyncSave();
    }

    private void scheduleAsyncSave() {
        if (this.savePending.compareAndSet(false, true)) {
            this.saveExecutor.schedule(() -> {
                this.savePending.set(false);
                this.saveData();
            }, 1L, TimeUnit.SECONDS);
        }
    }

    public boolean isOnline(String uuid) {
        return this.onlineSessions.containsKey(uuid);
    }

    public Map<String, Object> getSessionInfo(String uuid) {
        LinkedHashMap<String, Object> info = new LinkedHashMap<String, Object>();
        try {
            Long sessionStart = this.onlineSessions.get(uuid);
            if (sessionStart != null) {
                info.put("online", true);
                info.put("sessionStart", sessionStart);
                info.put("sessionDurationMs", System.currentTimeMillis() - sessionStart);
                PlayerData data = this.playerData.get(uuid);
                if (data != null && data.firstSeen() > 0L) {
                    info.put("firstSeen", data.firstSeen());
                }
                return info;
            }
            PlayerData data = this.playerData.get(uuid);
            if (data != null && data.lastSeen() > 0L) {
                info.put("online", false);
                info.put("lastSeen", data.lastSeen());
                if (data.firstSeen() > 0L) {
                    info.put("firstSeen", data.firstSeen());
                }
                if (data.lastSessionDurationMs() > 0L) {
                    info.put("lastSessionDurationMs", data.lastSessionDurationMs());
                }
                if (data.lastX() != null && data.lastY() != null && data.lastZ() != null) {
                    LinkedHashMap<String, Object> lastPosition = new LinkedHashMap<String, Object>();
                    lastPosition.put("x", Math.round(data.lastX()));
                    lastPosition.put("y", Math.round(data.lastY()));
                    lastPosition.put("z", Math.round(data.lastZ()));
                    if (data.lastDimension() != null) {
                        lastPosition.put("dimension", data.lastDimension());
                    }
                    info.put("lastPosition", lastPosition);
                }
                return info;
            }
            Path statsFile = this.server.getWorldPath(LevelResource.PLAYER_STATS_DIR).resolve(uuid + ".json");
            if (Files.exists(statsFile, new LinkOption[0])) {
                long mtime = Files.getLastModifiedTime(statsFile, new LinkOption[0]).toMillis();
                info.put("online", false);
                info.put("lastSeen", mtime);
                info.put("lastSeenSource", "file_mtime");
                return info;
            }
        }
        catch (Exception e) {
            LOGGER.debug("[Tracker] Error getting session info for {}: {}", (Object)uuid, (Object)e.getMessage());
        }
        info.put("online", false);
        info.put("lastSeen", null);
        return info;
    }

    public Map<String, Map<String, Object>> getAllSessionInfo() {
        LinkedHashMap<String, Map<String, Object>> all = new LinkedHashMap<String, Map<String, Object>>();
        for (String uuid : this.playerData.keySet()) {
            all.put(uuid, this.getSessionInfo(uuid));
        }
        try {
            Path statsDir = this.server.getWorldPath(LevelResource.PLAYER_STATS_DIR);
            if (Files.exists(statsDir, new LinkOption[0])) {
                Files.list(statsDir).filter(p -> p.toString().endsWith(".json")).forEach(p -> {
                    String uuid = p.getFileName().toString().replace(".json", "");
                    if (!all.containsKey(uuid)) {
                        all.put(uuid, this.getSessionInfo(uuid));
                    }
                });
            }
        }
        catch (IOException e) {
            LOGGER.error("[Tracker] Error listing stats directory", (Throwable)e);
        }
        return all;
    }

    private void saveData() {
        try {
            String json = this.gson.toJson(this.playerData);
            Files.writeString(this.dataFile, (CharSequence)json, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);
        }
        catch (IOException e) {
            LOGGER.error("[Tracker] Failed to save player tracking data", (Throwable)e);
        }
    }

    private void loadData() {
        if (!Files.exists(this.dataFile, new LinkOption[0])) {
            LOGGER.info("[Tracker] No existing player tracking data found");
            return;
        }
        try {
            String json = Files.readString(this.dataFile);
            Type type = new TypeToken<Map<String, PlayerData>>(this){}.getType();
            Map loaded = (Map)this.gson.fromJson(json, type);
            if (loaded != null) {
                loaded.forEach((uuid, data) -> this.playerData.put((String)uuid, new PlayerData(data.name(), data.lastSeen(), data.firstSeen(), 0L, data.lastSessionDurationMs(), data.lastX(), data.lastY(), data.lastZ(), data.lastDimension())));
            }
            LOGGER.info("[Tracker] Loaded tracking data for {} players", (Object)this.playerData.size());
        }
        catch (Exception e) {
            LOGGER.error("[Tracker] Failed to load player tracking data", (Throwable)e);
        }
    }

    public int getOnlineCount() {
        return this.onlineSessions.size();
    }

    public int getTrackedCount() {
        return this.playerData.size();
    }

    public void capturePositionForShutdown(String uuid, double x, double y, double z, String dimension) {
        this.playerData.computeIfPresent(uuid, (key, existing) -> new PlayerData(existing.name(), existing.lastSeen(), existing.firstSeen(), existing.loginTime(), existing.lastSessionDurationMs(), x, y, z, dimension));
    }

    public void shutdown() {
        long now = System.currentTimeMillis();
        for (Map.Entry<String, Long> entry : this.onlineSessions.entrySet()) {
            String uuid = entry.getKey();
            long sessionStart = entry.getValue();
            long duration = now - sessionStart;
            this.playerData.computeIfPresent(uuid, (key, existing) -> existing.withLastSeen(now, duration));
        }
        this.onlineSessions.clear();
        this.saveData();
        this.saveExecutor.shutdownNow();
        LOGGER.info("[Tracker] PlayerTracker shutdown complete");
    }

    public record PlayerData(String name, long lastSeen, long firstSeen, long loginTime, long lastSessionDurationMs, Double lastX, Double lastY, Double lastZ, String lastDimension) {
        public PlayerData(String name, long now) {
            this(name, 0L, now, now, 0L, null, null, null, null);
        }

        public PlayerData withLastSeen(long lastSeen, long sessionDuration) {
            return new PlayerData(this.name, lastSeen, this.firstSeen, 0L, sessionDuration, this.lastX, this.lastY, this.lastZ, this.lastDimension);
        }

        public PlayerData withLastSeen(long lastSeen, long sessionDuration, double x, double y, double z, String dimension) {
            return new PlayerData(this.name, lastSeen, this.firstSeen, 0L, sessionDuration, x, y, z, dimension);
        }

        public PlayerData withLogin(String name, long loginTime) {
            return new PlayerData(name, this.lastSeen, this.firstSeen, loginTime, this.lastSessionDurationMs, this.lastX, this.lastY, this.lastZ, this.lastDimension);
        }
    }
}

