/*
 * Decompiled with CFR 0.152.
 */
package com.crittafur.statsapi;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.google.gson.JsonSyntaxException;
import java.io.IOException;
import java.io.Serializable;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.time.OffsetDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AdvancementsParser {
    private static final Logger LOGGER = LoggerFactory.getLogger((String)"StatsAPI");
    static final long MAX_FILE_SIZE = 0xA00000L;
    private static final DateTimeFormatter MC_TIMESTAMP_FORMAT = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss Z");

    public static Map<String, Object> parseAdvancementsFile(Path file, boolean includePartial) throws IOException {
        if (!Files.exists(file, new LinkOption[0])) {
            return null;
        }
        long fileSize = Files.size(file);
        if (fileSize > 0xA00000L) {
            LOGGER.warn("[Advancements] File too large, skipping: {} ({} bytes)", (Object)file, (Object)fileSize);
            return null;
        }
        String content = Files.readString(file);
        return AdvancementsParser.parseAdvancementsJson(content, includePartial);
    }

    public static Map<String, Object> parseAdvancementsJson(String jsonContent, boolean includePartial) {
        JsonObject root;
        try {
            JsonElement parsed = JsonParser.parseString((String)jsonContent);
            if (!parsed.isJsonObject()) {
                LOGGER.warn("[Advancements] Content is not a JSON object");
                return null;
            }
            root = parsed.getAsJsonObject();
        }
        catch (JsonSyntaxException e) {
            LOGGER.warn("[Advancements] Invalid JSON syntax: {}", (Object)e.getMessage());
            return null;
        }
        Map<String, AdvancementEntry> filtered = AdvancementsParser.filterAdvancements(root);
        return AdvancementsParser.groupByCategory(filtered, includePartial);
    }

    static Map<String, AdvancementEntry> filterAdvancements(JsonObject root) {
        LinkedHashMap<String, AdvancementEntry> result = new LinkedHashMap<String, AdvancementEntry>();
        for (Map.Entry entry : root.entrySet()) {
            JsonElement value;
            String key = (String)entry.getKey();
            if (key.equals("DataVersion") || key.contains("/recipe/") || AdvancementsParser.isRecipeAdvancement(key) || AdvancementsParser.isRootAdvancement(key) || !(value = (JsonElement)entry.getValue()).isJsonObject()) continue;
            JsonObject advObj = value.getAsJsonObject();
            boolean done = advObj.has("done") && advObj.get("done").getAsBoolean();
            LinkedHashMap<String, Long> criteria = new LinkedHashMap<String, Long>();
            if (advObj.has("criteria") && advObj.get("criteria").isJsonObject()) {
                for (Map.Entry criterion : advObj.getAsJsonObject("criteria").entrySet()) {
                    if (!((JsonElement)criterion.getValue()).isJsonPrimitive()) continue;
                    long timestamp = AdvancementsParser.parseTimestamp(((JsonElement)criterion.getValue()).getAsString());
                    criteria.put((String)criterion.getKey(), timestamp);
                }
            }
            result.put(key, new AdvancementEntry(key, done, criteria));
        }
        return result;
    }

    static boolean isRecipeAdvancement(String key) {
        int colonIndex = key.indexOf(58);
        if (colonIndex >= 0) {
            String path = key.substring(colonIndex + 1);
            return path.startsWith("recipe/") || path.equals("recipe");
        }
        return false;
    }

    static boolean isRootAdvancement(String key) {
        if (key.endsWith(":root")) {
            return true;
        }
        int colonIndex = key.indexOf(58);
        if (colonIndex >= 0) {
            String path = key.substring(colonIndex + 1);
            return path.endsWith("/root");
        }
        return false;
    }

    static Map<String, Object> groupByCategory(Map<String, AdvancementEntry> entries, boolean includePartial) {
        LinkedHashMap<String, List> groups = new LinkedHashMap<String, List>();
        for (AdvancementEntry entry : entries.values()) {
            String categoryKey = AdvancementsParser.getCategoryKey(entry.id());
            groups.computeIfAbsent(categoryKey, k -> new ArrayList()).add(entry);
        }
        int totalCompleted = 0;
        int totalCount = 0;
        LinkedHashMap categories = new LinkedHashMap();
        for (Map.Entry group : groups.entrySet()) {
            String categoryName = (String)group.getKey();
            List categoryEntries = (List)group.getValue();
            int catCompleted = 0;
            int catTotal = categoryEntries.size();
            ArrayList<Map<String, Object>> list = new ArrayList<Map<String, Object>>();
            for (AdvancementEntry entry : categoryEntries) {
                if (entry.done()) {
                    ++catCompleted;
                }
                if (!entry.done() && !includePartial) continue;
                list.add(AdvancementsParser.buildAdvancementResponse(entry, includePartial));
            }
            totalCompleted += catCompleted;
            totalCount += includePartial ? catTotal : catCompleted;
            if (list.isEmpty()) continue;
            LinkedHashMap<String, Serializable> category = new LinkedHashMap<String, Serializable>();
            category.put("completed", Integer.valueOf(catCompleted));
            category.put("total", Integer.valueOf(includePartial ? catTotal : catCompleted));
            category.put("list", list);
            categories.put(categoryName, category);
        }
        LinkedHashMap<String, Object> result = new LinkedHashMap<String, Object>();
        result.put("completed", totalCompleted);
        result.put("total", totalCount);
        result.put("categories", categories);
        return result;
    }

    private static Map<String, Object> buildAdvancementResponse(AdvancementEntry entry, boolean includePartial) {
        LinkedHashMap<String, Object> response = new LinkedHashMap<String, Object>();
        response.put("id", entry.id());
        response.put("done", entry.done());
        if (entry.done() && !entry.criteria().isEmpty()) {
            long latestTimestamp = entry.criteria().values().stream().mapToLong(Long::longValue).max().orElse(0L);
            response.put("completed_at", latestTimestamp > 0L ? Long.valueOf(latestTimestamp) : null);
        } else {
            response.put("completed_at", null);
        }
        if (!entry.done() && includePartial && !entry.criteria().isEmpty()) {
            LinkedHashMap<String, Integer> progress = new LinkedHashMap<String, Integer>();
            progress.put("completed_criteria", entry.criteria().size());
            response.put("progress", progress);
        }
        return response;
    }

    static String getCategoryKey(String advancementId) {
        int colonIndex = advancementId.indexOf(58);
        if (colonIndex < 0) {
            return advancementId;
        }
        String namespace = advancementId.substring(0, colonIndex);
        String path = advancementId.substring(colonIndex + 1);
        int slashIndex = path.indexOf(47);
        if (slashIndex < 0) {
            return namespace;
        }
        String category = path.substring(0, slashIndex);
        return namespace + ":" + category;
    }

    static long parseTimestamp(String timestamp) {
        if (timestamp == null || timestamp.isEmpty()) {
            return 0L;
        }
        try {
            OffsetDateTime odt = OffsetDateTime.parse(timestamp, MC_TIMESTAMP_FORMAT);
            return odt.toInstant().toEpochMilli();
        }
        catch (DateTimeParseException e) {
            LOGGER.debug("[Advancements] Failed to parse timestamp: {}", (Object)timestamp);
            return 0L;
        }
    }

    public static Set<String> getCompletedAdvancementIds(String jsonContent) {
        JsonElement parsed;
        try {
            parsed = JsonParser.parseString((String)jsonContent);
        }
        catch (JsonSyntaxException e) {
            return Set.of();
        }
        if (!parsed.isJsonObject()) {
            return Set.of();
        }
        Map<String, AdvancementEntry> filtered = AdvancementsParser.filterAdvancements(parsed.getAsJsonObject());
        LinkedHashSet<String> completed = new LinkedHashSet<String>();
        for (AdvancementEntry entry : filtered.values()) {
            if (!entry.done()) continue;
            completed.add(entry.id());
        }
        return completed;
    }

    record AdvancementEntry(String id, boolean done, Map<String, Long> criteria) {
    }
}

